/*-
 * Copyright (c) 2003-2004 Tim Kientzle
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer
 *    in this position and unchanged.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR(S) ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR(S) BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/lib/libarchive/archive.h.in,v 1.31 2006/09/05 05:59:45 kientzle Exp $
 */

#ifndef ARCHIVE_H_INCLUDED
#define	ARCHIVE_H_INCLUDED

/*
 * This header file corresponds to:
 *   Library version 1.3.1
 *   Shared library version @SHLIB_MAJOR@
 */

#include <sys/types.h>  /* Linux requires this for off_t */
#include <inttypes.h> /* For int64_t */
#include <stdio.h> /* For FILE * */
#include <unistd.h>  /* For ssize_t and size_t */

#ifdef __cplusplus
extern "C" {
#endif


/*
 * If ARCHIVE_API_VERSION != archive_api_version(), then the library you
 * were linked with is using an incompatible API to the one you were
 * compiled with.  This is almost certainly a fatal problem.
 *
 * ARCHIVE_API_FEATURE is incremented with each significant feature
 * addition, so you can test (at compile or run time) if a particular
 * feature is implemented.  It's no big deal if ARCHIVE_API_FEATURE !=
 * archive_api_feature(), as long as both are high enough to include
 * the features you're relying on.  Specific values of FEATURE are
 * documented here:
 *
 *    1 - Version tests are available.
 *    2 - archive_{read,write}_close available separately from _finish.
 *    3 - open_memory, open_memory2, open_FILE, open_fd available
 */
#define	ARCHIVE_API_VERSION	1
int		archive_api_version(void);
#define	ARCHIVE_API_FEATURE	3
int		archive_api_feature(void);
/* Textual name/version of the library. */
#define	ARCHIVE_LIBRARY_VERSION	"libarchive 1.3.1"
const char *	archive_version(void);

#define	ARCHIVE_BYTES_PER_RECORD	  512
#define	ARCHIVE_DEFAULT_BYTES_PER_BLOCK	10240

/* Declare our basic types. */
struct archive;
struct archive_entry;

/*
 * Error codes: Use archive_errno() and archive_error_string()
 * to retrieve details.  Unless specified otherwise, all functions
 * that return 'int' use these codes.
 */
#define	ARCHIVE_EOF	  1	/* Found end of archive. */
#define	ARCHIVE_OK	  0	/* Operation was successful. */
#define	ARCHIVE_RETRY	(-10)	/* Retry might succeed. */
#define	ARCHIVE_WARN	(-20)	/* Partial sucess. */
#define	ARCHIVE_FATAL	(-30)	/* No more operations are possible. */

/*
 * As far as possible, archive_errno returns standard platform errno codes.
 * Of course, the details vary by platform, so the actual definitions
 * here are stored in "archive_platform.h".  The symbols are listed here
 * for reference; as a rule, clients should not need to know the exact
 * platform-dependent error code.
 */
/* Unrecognized or invalid file format. */
/* #define	ARCHIVE_ERRNO_FILE_FORMAT */
/* Illegal usage of the library. */
/* #define	ARCHIVE_ERRNO_PROGRAMMER_ERROR */
/* Unknown or unclassified error. */
/* #define	ARCHIVE_ERRNO_MISC */

/*
 * Callbacks are invoked to automatically read/skip/write/open/close the
 * archive. You can provide your own for complex tasks (like breaking
 * archives across multiple tapes) or use standard ones built into the
 * library.
 */

/* Returns pointer and size of next block of data from archive. */
typedef ssize_t	archive_read_callback(struct archive *, void *_client_data,
		    const void **_buffer);
/* Skips at most request bytes from archive and returns the skipped amount */
typedef ssize_t	archive_skip_callback(struct archive *, void *_client_data,
		    size_t request);
/* Returns size actually written, zero on EOF, -1 on error. */
typedef ssize_t	archive_write_callback(struct archive *, void *_client_data,
		    void *_buffer, size_t _length);
typedef int	archive_open_callback(struct archive *, void *_client_data);
typedef int	archive_close_callback(struct archive *, void *_client_data);

/*
 * Codes for archive_compression.
 */
#define	ARCHIVE_COMPRESSION_NONE	0
#define	ARCHIVE_COMPRESSION_GZIP	1
#define	ARCHIVE_COMPRESSION_BZIP2	2
#define	ARCHIVE_COMPRESSION_COMPRESS	3

/*
 * Codes returned by archive_format.
 *
 * Top 16 bits identifies the format family (e.g., "tar"); lower
 * 16 bits indicate the variant.  This is updated by read_next_header.
 * Note that the lower 16 bits will often vary from entry to entry.
 */
#define	ARCHIVE_FORMAT_BASE_MASK		0xff0000U
#define	ARCHIVE_FORMAT_CPIO			0x10000
#define	ARCHIVE_FORMAT_CPIO_POSIX		(ARCHIVE_FORMAT_CPIO | 1)
#define	ARCHIVE_FORMAT_CPIO_BIN_LE		(ARCHIVE_FORMAT_CPIO | 2)
#define	ARCHIVE_FORMAT_CPIO_BIN_BE		(ARCHIVE_FORMAT_CPIO | 3)
#define	ARCHIVE_FORMAT_CPIO_SVR4_NOCRC		(ARCHIVE_FORMAT_CPIO | 4)
#define	ARCHIVE_FORMAT_CPIO_SVR4_CRC		(ARCHIVE_FORMAT_CPIO | 5)
#define	ARCHIVE_FORMAT_SHAR			0x20000
#define	ARCHIVE_FORMAT_SHAR_BASE		(ARCHIVE_FORMAT_SHAR | 1)
#define	ARCHIVE_FORMAT_SHAR_DUMP		(ARCHIVE_FORMAT_SHAR | 2)
#define	ARCHIVE_FORMAT_TAR			0x30000
#define	ARCHIVE_FORMAT_TAR_USTAR		(ARCHIVE_FORMAT_TAR | 1)
#define	ARCHIVE_FORMAT_TAR_PAX_INTERCHANGE	(ARCHIVE_FORMAT_TAR | 2)
#define	ARCHIVE_FORMAT_TAR_PAX_RESTRICTED	(ARCHIVE_FORMAT_TAR | 3)
#define	ARCHIVE_FORMAT_TAR_GNUTAR		(ARCHIVE_FORMAT_TAR | 4)
#define	ARCHIVE_FORMAT_ISO9660			0x40000
#define	ARCHIVE_FORMAT_ISO9660_ROCKRIDGE	(ARCHIVE_FORMAT_ISO9660 | 1)
#define	ARCHIVE_FORMAT_ZIP			0x50000

/*-
 * Basic outline for reading an archive:
 *   1) Ask archive_read_new for an archive reader object.
 *   2) Update any global properties as appropriate.
 *      In particular, you'll certainly want to call appropriate
 *      archive_read_support_XXX functions.
 *   3) Call archive_read_open_XXX to open the archive
 *   4) Repeatedly call archive_read_next_header to get information about
 *      successive archive entries.  Call archive_read_data to extract
 *      data for entries of interest.
 *   5) Call archive_read_finish to end processing.
 */
struct archive	*archive_read_new(void);

/*
 * The archive_read_support_XXX calls enable auto-detect for this
 * archive handle.  They also link in the necessary support code.
 * For example, if you don't want bzlib linked in, don't invoke
 * support_compression_bzip2().  The "all" functions provide the
 * obvious shorthand.
 */
int		 archive_read_support_compression_all(struct archive *);
int		 archive_read_support_compression_bzip2(struct archive *);
int		 archive_read_support_compression_compress(struct archive *);
int		 archive_read_support_compression_gzip(struct archive *);
int		 archive_read_support_compression_none(struct archive *);

int		 archive_read_support_format_all(struct archive *);
int		 archive_read_support_format_cpio(struct archive *);
int		 archive_read_support_format_gnutar(struct archive *);
int		 archive_read_support_format_iso9660(struct archive *);
int		 archive_read_support_format_tar(struct archive *);
int		 archive_read_support_format_zip(struct archive *);


/* Open the archive using callbacks for archive I/O. */
int		 archive_read_open(struct archive *, void *_client_data,
		     archive_open_callback *, archive_read_callback *,
		     archive_close_callback *);
int		 archive_read_open2(struct archive *, void *_client_data,
		     archive_open_callback *, archive_read_callback *,
		     archive_skip_callback *, archive_close_callback *);

/*
 * A variety of shortcuts that invoke archive_read_open() with
 * canned callbacks suitable for common situations.  The ones that
 * accept a block size handle tape blocking correctly.
 */
/* Use this if you know the filename.  Note: NULL indicates stdin. */
int		 archive_read_open_filename(struct archive *,
		     const char *_filename, size_t _block_size);
/* archive_read_open_file() is a deprecated synonym for ..._open_filename(). */
int		 archive_read_open_file(struct archive *,
		     const char *_filename, size_t _block_size);
/* Read an archive that's stored in memory. */
int		 archive_read_open_memory(struct archive *,
		     void * buff, size_t size);
/* A more involved version that is only used for internal testing. */
int		archive_read_open_memory2(struct archive *a, void *buff,
		     size_t size, size_t read_size);
/* Read an archive that's already open, using the file descriptor. */
int		 archive_read_open_fd(struct archive *, int _fd,
		     size_t _block_size);
/* Read an archive that's already open, using a FILE *. */
/* Note: DO NOT use this with tape drives. */
int		 archive_read_open_FILE(struct archive *, FILE *_file);

/* Parses and returns next entry header. */
int		 archive_read_next_header(struct archive *,
		     struct archive_entry **);

/*
 * Retrieve the byte offset in UNCOMPRESSED data where last-read
 * header started.
 */
int64_t		 archive_read_header_position(struct archive *);

/* Read data from the body of an entry.  Similar to read(2). */
ssize_t		 archive_read_data(struct archive *, void *, size_t);
/*
 * A zero-copy version of archive_read_data that also exposes the file offset
 * of each returned block.  Note that the client has no way to specify
 * the desired size of the block.  The API does gaurantee that offsets will
 * be strictly increasing and that returned blocks will not overlap.
 */
int		 archive_read_data_block(struct archive *a,
		    const void **buff, size_t *size, off_t *offset);

/*-
 * Some convenience functions that are built on archive_read_data:
 *  'skip': skips entire entry
 *  'into_buffer': writes data into memory buffer that you provide
 *  'into_fd': writes data to specified filedes
 */
int		 archive_read_data_skip(struct archive *);
int		 archive_read_data_into_buffer(struct archive *, void *buffer,
		     ssize_t len);
int		 archive_read_data_into_fd(struct archive *, int fd);

/*-
 * Convenience function to recreate the current entry (whose header
 * has just been read) on disk.
 *
 * This does quite a bit more than just copy data to disk. It also:
 *  - Creates intermediate directories as required.
 *  - Manages directory permissions:  non-writable directories will
 *    be initially created with write permission enabled; when the
 *    archive is closed, dir permissions are edited to the values specified
 *    in the archive.
 *  - Checks hardlinks:  hardlinks will not be extracted unless the
 *    linked-to file was also extracted within the same session. (TODO)
 */

/* The "flags" argument selects optional behavior, 'OR' the flags you want. */
/* TODO: The 'Default' comments here are not quite correct; clean this up. */
#define	ARCHIVE_EXTRACT_OWNER	(1) /* Default: owner/group not restored */
#define	ARCHIVE_EXTRACT_PERM	(2) /* Default: restore perm only for reg file*/
#define	ARCHIVE_EXTRACT_TIME	(4) /* Default: mod time not restored */
#define	ARCHIVE_EXTRACT_NO_OVERWRITE (8) /* Default: Replace files on disk */
#define	ARCHIVE_EXTRACT_UNLINK	(16) /* Default: don't unlink existing files */
#define	ARCHIVE_EXTRACT_ACL	(32) /* Default: don't restore ACLs */
#define	ARCHIVE_EXTRACT_FFLAGS	(64) /* Default: don't restore fflags */
#define	ARCHIVE_EXTRACT_XATTR   (128) /* Default: don't restore xattrs */

int		 archive_read_extract(struct archive *, struct archive_entry *,
		     int flags);
void		 archive_read_extract_set_progress_callback(struct archive *,
		     void (*_progress_func)(void *), void *_user_data);

/* Record the dev/ino of a file that will not be written.  This is
 * generally set to the dev/ino of the archive being read. */
void		archive_read_extract_set_skip_file(struct archive *,
		     dev_t, ino_t);

/* Close the file and release most resources. */
int		 archive_read_close(struct archive *);
/* Release all resources and destroy the object. */
/* Note that archive_read_finish will call archive_read_close for you. */
#if ARCHIVE_API_VERSION > 1
int		 archive_read_finish(struct archive *);
#else
/* Temporarily allow library to compile with either 1.x or 2.0 API. */
/* Erroneously declared to return void in libarchive 1.x */
void		 archive_read_finish(struct archive *);
#endif

/*-
 * To create an archive:
 *   1) Ask archive_write_new for a archive writer object.
 *   2) Set any global properties.  In particular, you should set
 *      the compression and format to use.
 *   3) Call archive_write_open to open the file (most people
 *       will use archive_write_open_file or archive_write_open_fd,
 *       which provide convenient canned I/O callbacks for you).
 *   4) For each entry:
 *      - construct an appropriate struct archive_entry structure
 *      - archive_write_header to write the header
 *      - archive_write_data to write the entry data
 *   5) archive_write_close to close the output
 *   6) archive_write_finish to cleanup the writer and release resources
 */
struct archive	*archive_write_new(void);
int		 archive_write_set_bytes_per_block(struct archive *,
		     int bytes_per_block);
int		 archive_write_get_bytes_per_block(struct archive *);
/* XXX This is badly misnamed; suggestions appreciated. XXX */
int		 archive_write_set_bytes_in_last_block(struct archive *,
		     int bytes_in_last_block);
int		 archive_write_get_bytes_in_last_block(struct archive *);

/* The dev/ino of a file that won't be archived.  This is used
 * to avoid recursively adding an archive to itself. */
int		 archive_write_set_skip_file(struct archive *, dev_t, ino_t);

int		 archive_write_set_compression_bzip2(struct archive *);
int		 archive_write_set_compression_gzip(struct archive *);
int		 archive_write_set_compression_none(struct archive *);
/* A convenience function to set the format based on the code or name. */
int		 archive_write_set_format(struct archive *, int format_code);
int		 archive_write_set_format_by_name(struct archive *,
		     const char *name);
/* To minimize link pollution, use one or more of the following. */
int		 archive_write_set_format_cpio(struct archive *);
/* TODO: int archive_write_set_format_old_tar(struct archive *); */
int		 archive_write_set_format_pax(struct archive *);
int		 archive_write_set_format_pax_restricted(struct archive *);
int		 archive_write_set_format_shar(struct archive *);
int		 archive_write_set_format_shar_dump(struct archive *);
int		 archive_write_set_format_ustar(struct archive *);
int		 archive_write_open(struct archive *, void *,
		     archive_open_callback *, archive_write_callback *,
		     archive_close_callback *);
int		 archive_write_open_fd(struct archive *, int _fd);
int		 archive_write_open_filename(struct archive *, const char *_file);
/* A deprecated synonym for archive_write_open_filename() */
int		 archive_write_open_file(struct archive *, const char *_file);
int		 archive_write_open_FILE(struct archive *, FILE *);
/* _buffSize is the size of the buffer, _used refers to a variable that
 * will be updated after each write into the buffer. */
int		 archive_write_open_memory(struct archive *,
			void *_buffer, size_t _buffSize, size_t *_used);

/*
 * Note that the library will truncate writes beyond the size provided
 * to archive_write_header or pad if the provided data is short.
 */
int		 archive_write_header(struct archive *,
		     struct archive_entry *);
#if ARCHIVE_API_VERSION > 1
ssize_t		 archive_write_data(struct archive *, const void *, size_t);
#else
/* Temporarily allow library to compile with either 1.x or 2.0 API. */
/* This was erroneously declared to return "int" in libarchive 1.x. */
int		 archive_write_data(struct archive *, const void *, size_t);
#endif
int		 archive_write_close(struct archive *);
#if ARCHIVE_API_VERSION > 1
int		 archive_write_finish(struct archive *);
#else
/* Temporarily allow library to compile with either 1.x or 2.0 API. */
/* Return value was incorrect in libarchive 1.x. */
void		 archive_write_finish(struct archive *);
#endif

/*
 * Accessor functions to read/set various information in
 * the struct archive object:
 */
/* Bytes written after compression or read before decompression. */
int64_t		 archive_position_compressed(struct archive *);
/* Bytes written to compressor or read from decompressor. */
int64_t		 archive_position_uncompressed(struct archive *);

const char	*archive_compression_name(struct archive *);
int		 archive_compression(struct archive *);
int		 archive_errno(struct archive *);
const char	*archive_error_string(struct archive *);
const char	*archive_format_name(struct archive *);
int		 archive_format(struct archive *);
void		 archive_set_error(struct archive *, int _err, const char *fmt, ...);

#ifdef __cplusplus
}
#endif

#endif /* !ARCHIVE_H_INCLUDED */
